import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Languages, TrendingUp, Star, BookOpen, Award, Globe, ChevronRight, Calendar } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyCourses.css'

const MyCourses = () => {
  const [courses, setCourses] = useState([])

  useEffect(() => {
    loadCourses()
    const interval = setInterval(loadCourses, 500)
    return () => clearInterval(interval)
  }, [])

  const getIconComponent = (iconType) => {
    const icons = {
      'globe': Globe,
      'languages': Languages,
      'book': BookOpen,
      'award': Award
    }
    const IconComponent = icons[iconType] || Globe
    return <IconComponent size={28} strokeWidth={2} />
  }

  const loadCourses = () => {
    const saved = localStorage.getItem('languageCourses')
    if (saved) {
      try {
        const parsed = JSON.parse(saved)
        setCourses(parsed)
      } catch (e) {
        setCourses([])
      }
    } else {
      const defaultCourses = [
        {
          id: 1,
          language: 'Английский',
          level: 'Intermediate',
          progress: 65,
          rating: 5,
          iconType: 'globe'
        },
        {
          id: 2,
          language: 'Испанский',
          level: 'Beginner',
          progress: 30,
          rating: 4,
          iconType: 'languages'
        },
        {
          id: 3,
          language: 'Китайский',
          level: 'Advanced',
          progress: 85,
          rating: 5,
          iconType: 'book'
        }
      ]
      setCourses(defaultCourses)
      localStorage.setItem('languageCourses', JSON.stringify(defaultCourses))
    }
  }

  const getLanguageColor = (language) => {
    const colors = {
      'Английский': '#3B82F6',
      'Испанский': '#EF4444',
      'Китайский': '#F59E0B',
      'Немецкий': '#10B981'
    }
    return colors[language] || '#6366F1'
  }

  const renderStars = (rating) => {
    return Array.from({ length: 5 }, (_, i) => (
      <Star
        key={i}
        size={14}
        fill={i < rating ? '#A78BFA' : 'none'}
        stroke={i < rating ? '#A78BFA' : 'var(--text-light)'}
        strokeWidth={1.5}
      />
    ))
  }

  const averageRating = courses.length > 0 
    ? (courses.reduce((sum, c) => sum + c.rating, 0) / courses.length).toFixed(1)
    : 0

  const totalProgress = courses.length > 0
    ? Math.round(courses.reduce((sum, c) => sum + c.progress, 0) / courses.length)
    : 0

  return (
    <div className="courses-timeline-page">
      <motion.div 
        className="page-top-section"
        initial={{ opacity: 0, y: -30 }}
        animate={{ opacity: 1, y: 0 }}
        transition={{ duration: 0.6 }}
      >
        <div className="top-header">
          <h1 className="page-main-title">Мои курсы</h1>
          <p className="page-main-subtitle">История вашего обучения</p>
        </div>
        <div className="top-stats">
          <div className="stat-minimal">
            <div className="stat-minimal-value">{courses.length}</div>
            <div className="stat-minimal-label">Курсов</div>
          </div>
          <div className="stat-minimal">
            <div className="stat-minimal-value">{totalProgress}%</div>
            <div className="stat-minimal-label">Прогресс</div>
          </div>
          <div className="stat-minimal">
            <div className="stat-minimal-value">{averageRating}</div>
            <div className="stat-minimal-label">Рейтинг</div>
          </div>
        </div>
      </motion.div>

      <ImageSlider images={[
        '/images/language-1.jpg',
        '/images/language-2.jpg',
        '/images/language-3.jpg',
        '/images/language-4.jpg'
      ]} />

      <motion.section 
        className="timeline-section"
        initial={{ opacity: 0 }}
        animate={{ opacity: 1 }}
        transition={{ duration: 0.6, delay: 0.3 }}
      >
        {courses.length > 0 ? (
          <div className="timeline-wrapper">
            {courses.map((course, index) => (
              <motion.div 
                key={course.id} 
                className="timeline-item"
                initial={{ opacity: 0, x: -50 }}
                animate={{ opacity: 1, x: 0 }}
                transition={{ delay: index * 0.1, duration: 0.5 }}
              >
                <div className="timeline-marker" style={{ borderColor: getLanguageColor(course.language) }}>
                  <div className="marker-dot" style={{ background: getLanguageColor(course.language) }}>
                    {getIconComponent(course.iconType || 'globe')}
                  </div>
                </div>
                <div className="timeline-content">
                  <div className="timeline-card">
                    <div className="timeline-card-header">
                      <div className="timeline-title-section">
                        <h3 className="timeline-language">{course.language}</h3>
                        <span className="timeline-level" style={{ color: getLanguageColor(course.language) }}>
                          {course.level}
                        </span>
                      </div>
                      <div className="timeline-rating">
                        {renderStars(course.rating)}
                      </div>
                    </div>
                    <div className="timeline-progress-section">
                      <div className="timeline-progress-header">
                        <span className="timeline-progress-text">Прогресс обучения</span>
                        <span className="timeline-progress-value">{course.progress}%</span>
                      </div>
                      <div className="timeline-progress-bar">
                        <motion.div 
                          className="timeline-progress-fill"
                          initial={{ width: 0 }}
                          animate={{ width: `${course.progress}%` }}
                          transition={{ duration: 1, delay: index * 0.1 + 0.3 }}
                          style={{ background: getLanguageColor(course.language) }}
                        />
                      </div>
                    </div>
                    <div className="timeline-footer">
                      <div className="timeline-date">
                        <Calendar size={16} strokeWidth={2} />
                        <span>Начато: {new Date().toLocaleDateString('ru-RU', { month: 'long', year: 'numeric' })}</span>
                      </div>
                      <ChevronRight size={20} strokeWidth={2} className="timeline-arrow" />
                    </div>
                  </div>
                </div>
              </motion.div>
            ))}
          </div>
        ) : (
          <div className="empty-timeline">
            <div className="empty-timeline-icon">
              <BookOpen size={56} strokeWidth={1.5} />
            </div>
            <h3 className="empty-timeline-title">Нет активных курсов</h3>
            <p className="empty-timeline-text">Начните изучение нового языка</p>
          </div>
        )}
      </motion.section>
    </div>
  )
}

export default MyCourses
